--[[
Dump global functions, tables, classes and variables to a file. The purpose is to better understand the Farming Simulator object model through reverse engineering.

With this a starting point, you can then use the console command 'dtSaveTable' from the mod 'PowerTools: Developer' to write whole Lua tables (and full table hierarchies) to file for further analysis.

Author:     w33zl
Version:    1.2.0
Modified:   2024-11-20

Changelog:

]]

DataDump = Mod:init()

DataDump:source("lib/DevHelper.lua")

local OUTPUT_HEADER = [[
-- This file was automatically generated by the mod FS25 Data Dump (https://github.com/w33zl/FS25_DataDump)
]]

local g_powerTools


local function saveOutputToFile(basePath, name, table, maxDepth, omitPrefix)
    maxDepth = maxDepth or 1
    omitPrefix = omitPrefix or false
    local filePath = basePath .. name .. ".lua"
    if fileExists(filePath) then
        deleteFile(filePath)
    end
    local tableName = (omitPrefix and "" or "global") .. name
    g_powerTools:saveTable(tableName, table, filePath, maxDepth, nil, OUTPUT_HEADER)

    if not fileExists(filePath) then
        Log:error("Failed to save '%s' to '%s'", name, filePath)
    end

end

validatePowerToolsMod = function()
    if g_powerTools == nil then
        Log:warning("g_powerTools was not found, verify that the mod 'Developer PowerTools' is enabled.")
        return
    end

    return true
end

local function buildFilteredTable(startTable, maxDepth, filter, deepFilter)
    deepFilter = deepFilter or false
    maxDepth = maxDepth or 1
    

    local outputTable = {}

    local count = 0
    local function traverseTable(currentTable, depth, filter, deepFilter)
        depth = (depth or 0) + 1
        -- Log:var("depth", depth)
        -- Log:var("maxDepth", maxDepth)
        local newTable = {}

        local last = nil
        local doFilter = ( depth == 1 ) or deepFilter
        
        if depth > maxDepth then
            -- Log:debug("depth > maxDepth")
            -- Log:var("depth", depth)
            -- Log:var("maxDepth", maxDepth)
            return newTable
        end
        
        while true do
            count = count + 1
            local index, value = next(currentTable, last)
            if index ~= nil then
                if not doFilter or string.startsWith(index, filter) then
                    if type(value) == "table" then
                        newTable[index] = traverseTable(value, depth, filter, deepFilter)
                    else
                        newTable[index] = value
                    end
                else
                    -- print("Skipped " .. depth .. ": " .. index)
                end

                last = index
            else
                break
            end
        end
        return newTable
    end

    outputTable = traverseTable(startTable, nil, filter, deepFilter)

    return outputTable
end


function DataDump:loadMap(filename)
    g_powerTools = g_globalMods["FS25_PowerTools"]
    createFolder(g_currentModSettingsDirectory)
end

function DataDump:startMission()
    -- DataDump:consoleCommandDump()
end

function DataDump:consoleCommandDumpGlobalTables(maxDepth, forceDeepScan)
    if not validatePowerToolsMod() then
        return
    end

    local DEFAULT_MAX_DEPTH = 2
    local MAX_RECOMMENDED_DEPTH = 2
    local basePath = g_currentModSettingsDirectory

    maxDepth = tonumber(maxDepth) or DEFAULT_MAX_DEPTH
    forceDeepScan = forceDeepScan == "true"

    if not forceDeepScan and maxDepth > MAX_RECOMMENDED_DEPTH then
        Log:warning("Max depth is set to %d, recommended is maximum %d, please consider using a lower value or set to forced mode ('ddSaveGlobalTables %d true')", maxDepth, MAX_RECOMMENDED_DEPTH, maxDepth)
        return
    end

    local executionTimer = DevHelper.measureStart("Exporting g_currentMission and all global g_Xxxx tables %.2f seconds")

    xpcall(function()
        local gTables = buildFilteredTable(self.__g, maxDepth, "g_", false)
        saveOutputToFile(basePath, "g_tables", gTables, maxDepth, true)
    end, function(err) Log:error("Failed to export g_Xxxx tables: %s", err) end)
    
    xpcall(function()
        saveOutputToFile(basePath, "g_currentMission", g_currentMission, maxDepth, true)
    end, function(err) Log:error("Failed to export g_currentMission: %s", err) end)

    xpcall(function()
        saveOutputToFile(basePath, "g_localPlayer", g_localPlayer, maxDepth + 1, true)
    end, function(err) Log:error("Failed to export g_localPlayer: %s", err) end)

    executionTimer:stop(true)
    
end

function DataDump:consoleCommandDump(visualize, visualizeDepth)

    if self.triggerProcess or self.inProgress or self.isFinalizing then
        Log:warning("Dumping already in progress")
        return
    end

    

    self.executionTimer = DevHelper.measureStart("Processing global table took %.2f seconds")
    self.chunkTimer = DevHelper.measureStart()
    self.activeTable = self.__g
    self.triggerProcess = true
    self.chunkCount = 0
    self.output = {
        functions = {},
        classes = {},
        tables = {},
        fields = {}
    }
    self.stats = {
        functions = 0,
        classes = 0,
        tables = 0,
        fields = 0,
        total = 0,
    }
    self.visualize = visualize and true
    self.visualizeDepth = tonumber(visualizeDepth) or 2

    Log:debug("Visualize: %s, Depth: %d", tostring(self.visualize), self.visualizeDepth)
end

function DataDump:processChunk()
    --NOTE: Yes, this is over engineered, but it is prepared to handle a large number of tables in a deep structure
    local count = 0
    self.chunkCount = self.chunkCount + 1
    while true do
        count = count + 1
        local index, value = next(self.activeTable, self.last)
        self.last = index

        if self.last ~= nil then
            -- print(self.last)
            self.stats.total = self.stats.total + 1

            if type(value) == "function" then
                -- table.insert(self.output.functions, self.last)
                self.output.functions[self.last] = value
                self.stats.functions = self.stats.functions + 1
            elseif type(value) == "table" then
                local isClass = false
                if self.last == "StringUtil" or self.last == "g_splitTypeManager" then --HACK: Dirty solution to prevent callstack "error" due to StringUtil being obsolete
                    isClass = true
                elseif value.isa ~= nil and type(value.isa) == "function" then
                    isClass = value:isa(value) -- Should only be true on the actual class, but not on derived objects
                end

                if isClass then
                    self.output.classes[self.last] = value
                    self.stats.classes = self.stats.classes + 1
                else
                    self.output.tables[self.last] = value
                    self.stats.tables = self.stats.tables + 1
                end
            elseif type(value) == "userdata" then
                --TODO: need special care?
                self.output.fields[self.last] = value
                self.stats.fields = self.stats.fields + 1
            else
                self.output.fields[self.last] = value
                self.stats.fields = self.stats.fields + 1
            end
        end

        if self.last == nil or (self.chunkTimer:elapsed() > 1) or (count >= 5000) then
            count = 0
            self.chunkTimer = DevHelper.measureStart()
            return self.last
        end
    end
end

function DataDump:finalize()
    local basePath = g_currentModSettingsDirectory .. "global"
    local saveTimer = DevHelper.measureStart("Files saved in %.2f seconds")
    
    saveOutputToFile(basePath, "Functions", self.output.functions)
    saveOutputToFile(basePath, "Classes", self.output.classes)
    saveOutputToFile(basePath, "Tables", self.output.tables)
    saveOutputToFile(basePath, "Variables", self.output.fields)

    Log:info(saveTimer:stop(true))

    if self.visualize then
        g_powerTools:visualizeTable("Output", self.output, self.visualizeDepth)
    end

    self.isFinalizing = false
end

function DataDump:update(dt)
    if self.isFinalizing then
        DataDump:finalize()
        return
    elseif not self.triggerProcess and not self.inProgress then
        return
    end

    self.triggerProcess = false
    self.inProgress = true

    local val = self:processChunk()
    if val == nil then
        self.inProgress = false
        Log:info(self.executionTimer:stop(true))

        Log:info("Found %d functions, %d classes, %d tables and %d fields in %d chunks", self.stats.functions, self.stats.classes, self.stats.tables, self.stats.fields, self.chunkCount)

        if not validatePowerToolsMod() then
            return
        end

        Log:info("Saving output tables...")

        self.isFinalizing = true

        return
    else
        Log:info("#%d: Reading global table, found %d items so far... ", self.chunkCount, self.stats.total)
    end
end


addConsoleCommand("ddDump", "", "consoleCommandDump", DataDump)
addConsoleCommand("ddSaveGlobalTables", "", "consoleCommandDumpGlobalTables", DataDump)--TODO fix this!!!


