-- ============================================================================
-- === AutoGateSettings.Lua
-- === Mod by [LSMT] Modding Team 
-- === LS25 /FS25
-- === Script by [LSMT] BaTt3RiE @ 2025
-- === Ver 1.0.2.0
-- ============================================================================

AutoGateSettings = {}
AutoGateSettings.SETTINGS = {}
AutoGateSettings.CONTROLS = {}

local DEBUG = false
local function dprint(...) if DEBUG then print("[AutoGateAktivator]", ...) end end

AutoGateSettingsEvent = {}
AutoGateSettingsEvent_mt = Class(AutoGateSettingsEvent, Event)

InitEventClass(AutoGateSettingsEvent, "AutoGateSettingsEvent")

function AutoGateSettingsEvent.emptyNew()
    local self = Event.new(AutoGateSettingsEvent_mt)
    return self
end

function AutoGateSettingsEvent.new(closeDelay, checkOwnership)
    local self = AutoGateSettingsEvent.emptyNew()
    self.closeDelay = closeDelay
    self.checkOwnership = checkOwnership
    return self
end

function AutoGateSettingsEvent:readStream(streamId, connection)
    self.closeDelay = streamReadInt32(streamId)
    self.checkOwnership = streamReadBool(streamId)
    self:run(connection)
end

function AutoGateSettingsEvent:writeStream(streamId, connection)
    streamWriteInt32(streamId, self.closeDelay)
    streamWriteBool(streamId, self.checkOwnership)
end

function AutoGateSettingsEvent:run(connection)
    -- Wenn Client: Einstellungen vom Server übernehmen
    if not connection:getIsServer() then
        -- WICHTIG: Ignoriere Event wenn wir gerade selbst eine Änderung durchführen
        if AutoGateSettings.isChangingSettings then
            AutoGateSettings.isChangingSettings = false
            return
        end
        
        dprint("AutoGate: Einstellungen vom Server empfangen - Verzögerung: " .. self.closeDelay .. 
              "ms, Besitzprüfung: " .. tostring(self.checkOwnership))
              
        -- Lokale Einstellungen ohne zu speichern aktualisieren
        AutoGateSettings.closeDelay = self.closeDelay
        AutoGateSettings.checkOwnership = self.checkOwnership
        
        -- Globale Variablen für das Hauptskript aktualisieren
        CLOSE_DELAY_MS = self.closeDelay
        CHECK_OWNERSHIP = self.checkOwnership
        
        -- UI aktualisieren, falls Menü geöffnet ist
        local menu = g_gui.screenControllers[InGameMenu]
        if menu ~= nil and menu.isOpen then
            for _, control in pairs(AutoGateSettings.CONTROLS or {}) do
                if control.id == "closeDelay" then
                    local stateIndex = AutoGateSettings.getStateIndex("closeDelay")
                    control:setState(stateIndex)
                elseif control.id == "checkOwnership" then
                    local stateIndex = AutoGateSettings.getStateIndex("checkOwnership")
                    control:setState(stateIndex)
                end
            end
        end
    else
        -- Server empfängt Einstellungen von einem Client
        dprint("AutoGate (Server): Neue Einstellungen empfangen - Verzögerung: " .. self.closeDelay .. 
              "ms, Besitzprüfung: " .. tostring(self.checkOwnership))
              
        -- Lokale Einstellungen aktualisieren
        AutoGateSettings.closeDelay = self.closeDelay
        AutoGateSettings.checkOwnership = self.checkOwnership
        
        -- Globale Variablen für das Hauptskript aktualisieren
        CLOSE_DELAY_MS = self.closeDelay
        CHECK_OWNERSHIP = self.checkOwnership
        
        -- An alle anderen Clients weiterleiten
        AutoGateSettings.sendSettingsToClients()
    end
end

-- Verzögerungszeit-Optionen
AutoGateSettings.SETTINGS.closeDelay = {
    ['default'] = 3, -- Index des Standardwerts (10 Sekunden)
    ['values'] = {0, 5000, 10000, 20000, 30000, 60000, 120000, 300000},
    ['strings'] = {
        g_i18n:getText("autogate_setting_time_now"),
        g_i18n:getText("autogate_setting_time_5000"),
        g_i18n:getText("autogate_setting_time_10000"),
        g_i18n:getText("autogate_setting_time_20000"),
        g_i18n:getText("autogate_setting_time_30000"),
        g_i18n:getText("autogate_setting_time_60000"),
        g_i18n:getText("autogate_setting_time_120000"),
        g_i18n:getText("autogate_setting_time_300000")
    }
}

-- Besitzprüfung-Optionen
AutoGateSettings.SETTINGS.checkOwnership = {
    ['default'] = 1,  -- Index des Standardwerts (true)
    ['values'] = {true, false},
    ['strings'] = {
        g_i18n:getText("autogate_setting_onwership_true"),
        g_i18n:getText("autogate_setting_onwership_false")
    }
}

-- Standardwerte (werden später aus XML geladen)
AutoGateSettings.closeDelay = 0     -- 10 Sekunden
AutoGateSettings.checkOwnership = false  -- Besitzprüfung aktiviert

-- Sende Einstellungen an alle Clients
function AutoGateSettings.sendSettingsToClients()
    if g_server ~= nil then
        -- Event für alle Clients erstellen und senden
        g_server:broadcastEvent(AutoGateSettingsEvent.new(
            AutoGateSettings.closeDelay, 
            AutoGateSettings.checkOwnership
        ))
        dprint("AutoGate: Einstellungen an alle Clients gesendet")
        
        -- Auch auf dem Server selbst aktualisieren (wichtig!)
        CLOSE_DELAY_MS = AutoGateSettings.closeDelay
        CHECK_OWNERSHIP = AutoGateSettings.checkOwnership
    end
end

function AutoGateSettings.sendSettingsToServer()
    if g_client ~= nil and not g_currentMission:getIsServer() then
        -- Client sendet Änderungen an Server
        g_client:getServerConnection():sendEvent(AutoGateSettingsEvent.new(
            AutoGateSettings.closeDelay, 
            AutoGateSettings.checkOwnership
        ))
        dprint("AutoGate: Einstellungen an Server gesendet")
    end
end

-- Setze einen Wert
function AutoGateSettings.setValue(id, value)
    AutoGateSettings[id] = value
    
    -- Globale Variable aktualisieren, die vom Hauptscript verwendet wird
    if id == "closeDelay" then
        CLOSE_DELAY_MS = value
        dprint("AutoGate: Schließverzögerung auf " .. value .. "ms gesetzt")
    elseif id == "checkOwnership" then
        CHECK_OWNERSHIP = value
        dprint("AutoGate: Besitzprüfung " .. (value and "aktiviert" or "deaktiviert"))
    end
end

-- Hole einen Wert
function AutoGateSettings.getValue(id)
    return AutoGateSettings[id]
end

-- Finde den Index für einen bestimmten Wert
function AutoGateSettings.getStateIndex(id, value)
    local value = value or AutoGateSettings.getValue(id) 
    local values = AutoGateSettings.SETTINGS[id].values
    
    if type(value) == 'number' then
        for i, v in ipairs(values) do
            if v == value then
                return i
            end
        end
        
        -- Wenn kein exakter Treffer, nehme den nächstliegenden Wert
        local index = AutoGateSettings.SETTINGS[id].default
        local initialdiff = math.huge
        for i, v in ipairs(values) do
            local currentdiff = math.abs(v - value)
            if currentdiff < initialdiff then
                initialdiff = currentdiff
                index = i
            end 
        end
        return index
    elseif type(value) == 'boolean' then
        return value and 1 or 2
    end
    
    return AutoGateSettings.SETTINGS[id].default
end

-- Controls zum Einstellen der Einstellungen
AutoGateControls = {}

function AutoGateControls:onMenuOptionChanged(state, menuOption)
    
    local id = menuOption.id
    local setting = AutoGateSettings.SETTINGS
    local value = setting[id].values[state]
    
    if value ~= nil then
        AutoGateSettings.setValue(id, value)
        
        -- WICHTIG: Setze eine Flagge, dass wir gerade eine Änderung vornehmen
        AutoGateSettings.isChangingSettings = true
        
        -- Einstellungen speichern
        AutoGateSettings.saveSettings()
        
        -- Synchronisieren basierend auf Serverrolle
        if g_currentMission:getIsServer() then
            -- Als Server: Sende an alle Clients
            AutoGateSettings.sendSettingsToClients()
        else
            -- Als Client: Sende an Server
            AutoGateSettings.sendSettingsToServer()
        end
    end
end

-- Einstellungen aus XML-Datei laden
function AutoGateSettings.loadSettings()
    local xmlFilePath = Utils.getFilename("modSettings/AutoGateSettings.xml", getUserProfileAppPath())
    
    if not fileExists(xmlFilePath) then
        return
    end
    
    local xmlFile = loadXMLFile("AutoGateXML", xmlFilePath)
    if xmlFile == 0 then
        return
    end
    
    -- Verzögerungszeit laden
    local closeDelay = getXMLInt(xmlFile, "autoGate.settings#closeDelay")
    if closeDelay ~= nil then
        AutoGateSettings.setValue("closeDelay", closeDelay)
        dprint("AutoGate: Verzögerungszeit " .. closeDelay .. "ms geladen")
    end
    
    -- Besitzprüfung laden
    local checkOwnership = getXMLBool(xmlFile, "autoGate.settings#checkOwnership")
    if checkOwnership ~= nil then
        AutoGateSettings.setValue("checkOwnership", checkOwnership)
        dprint("AutoGate: Besitzprüfung " .. (checkOwnership and "aktiviert" or "deaktiviert"))
    end
    
    delete(xmlFile)
end

-- Einstellungen in XML-Datei speichern
function AutoGateSettings.saveSettings()
    local xmlFilePath = Utils.getFilename("modSettings/AutoGateSettings.xml", getUserProfileAppPath())
    local xmlFile = nil
    
    createFolder(getUserProfileAppPath() .. "modSettings/")
    
    if fileExists(xmlFilePath) then
        xmlFile = loadXMLFile("AutoGateXML", xmlFilePath)
    else
        xmlFile = createXMLFile("AutoGateXML", xmlFilePath, "autoGate")
    end
    
    if xmlFile == 0 then
        print("AutoGate: Fehler beim Öffnen der XML-Datei.")
        return
    end
    
    -- Verzögerungszeit speichern
    local closeDelay = AutoGateSettings.closeDelay
    setXMLInt(xmlFile, "autoGate.settings#closeDelay", closeDelay)
    
    -- Besitzprüfung speichern
    local checkOwnership = AutoGateSettings.checkOwnership
    setXMLBool(xmlFile, "autoGate.settings#checkOwnership", checkOwnership)
    
    saveXMLFile(xmlFile)
    delete(xmlFile)

    dprint("AutoGate: Einstellungen gespeichert - Verzögerungszeit: " .. closeDelay .. 
          "ms, Besitzprüfung: " .. (checkOwnership and "aktiviert" or "deaktiviert"))
end

-- Helfer-Funktion für FocusManager
local function updateFocusIds(element)
    if not element then
        return
    end
    element.focusId = FocusManager:serveAutoFocusId()
    for _, child in pairs(element.elements or {}) do
        updateFocusIds(child)
    end
end

-- Einstellungen ins InGame-Menü einfügen
function AutoGateSettings.injectMenu()
    local inGameMenu = g_gui.screenControllers[InGameMenu]
    if not inGameMenu then
        print("AutoGate: InGameMenu nicht gefunden!")
        return
    end
    
    local settingsPage = inGameMenu.pageSettings
    if not settingsPage then
        print("AutoGate: Settings-Page nicht gefunden!")
        return
    end
    
    local layoutToUse = settingsPage.gameSettingsLayout
    if not layoutToUse then
        print("AutoGate: gameSettingsLayout nicht gefunden!")
        return
    end
    
    -- Sektionsüberschrift erstellen
    local sectionTitle = nil
    for _, elem in ipairs(layoutToUse.elements) do
        if elem.name == "sectionHeader" then
            sectionTitle = elem:clone(layoutToUse)
            break
        end
    end
    
    if sectionTitle then
        sectionTitle:setText(g_i18n:getText("autogate_menu_section_title"))
    else
        sectionTitle = TextElement.new()
        sectionTitle:applyProfile("fs25_settingsSectionHeader", true)
        sectionTitle:setText(g_i18n:getText("autogate_menu_section_title"))
        sectionTitle.name = "sectionHeader"
        layoutToUse:addElement(sectionTitle)
    end
    
    sectionTitle.focusId = FocusManager:serveAutoFocusId()
    table.insert(settingsPage.controlsList, sectionTitle)
    AutoGateSettings.CONTROLS["sectionHeader"] = sectionTitle
    
    -- Verzögerungseinstellung erstellen
    local originalBox = settingsPage.multiVolumeVoiceBox
    if not originalBox then
        print("AutoGate: multiVolumeVoiceBox nicht gefunden!")
        return
    end
    
    -- 1. Verzögerungszeit-Einstellung
    local delayBox = originalBox:clone(layoutToUse)
    delayBox.id = "closeDelayBox"
    
    local delayOption = delayBox.elements[1]
    delayOption.id = "closeDelay"
    delayOption.target = AutoGateControls
    
    delayOption:setCallback("onClickCallback", "onMenuOptionChanged")
    delayOption:setDisabled(false)
    
    local toolTip = delayOption.elements[1]
    toolTip:setText(g_i18n:getText("autogate_menu_close_delay_tooltip"))
    delayBox.elements[2]:setText(g_i18n:getText("autogate_menu_close_delay"))
    
    delayOption:setTexts(AutoGateSettings.SETTINGS.closeDelay.strings)
    
    local delayStateIndex = AutoGateSettings.getStateIndex("closeDelay")
    delayOption:setState(delayStateIndex)
    
    AutoGateSettings.CONTROLS["closeDelay"] = delayOption
    
    updateFocusIds(delayBox)
    table.insert(settingsPage.controlsList, delayBox)
    
    -- 2. Besitzprüfung-Einstellung
    local ownershipBox = originalBox:clone(layoutToUse)
    ownershipBox.id = "checkOwnershipBox"
    
    local ownershipOption = ownershipBox.elements[1]
    ownershipOption.id = "checkOwnership"
    ownershipOption.target = AutoGateControls
    
    ownershipOption:setCallback("onClickCallback", "onMenuOptionChanged")
    ownershipOption:setDisabled(false)
    
    local ownershipToolTip = ownershipOption.elements[1]
    ownershipToolTip:setText(g_i18n:getText("autogate_menu_ownership_tooltip"))
    ownershipBox.elements[2]:setText(g_i18n:getText("autogate_menu_ownership"))
    
    ownershipOption:setTexts(AutoGateSettings.SETTINGS.checkOwnership.strings)
    
    local ownershipStateIndex = AutoGateSettings.getStateIndex("checkOwnership")
    ownershipOption:setState(ownershipStateIndex)
    
    AutoGateSettings.CONTROLS["checkOwnership"] = ownershipOption
    
    updateFocusIds(ownershipBox)
    table.insert(settingsPage.controlsList, ownershipBox)
    
    layoutToUse:invalidateLayout()
end

-- Initialisieren, wenn das Spiel geladen ist
Mission00.loadMission00Finished = Utils.appendedFunction(Mission00.loadMission00Finished, function()
    AutoGateSettings.loadSettings()
    AutoGateSettings.injectMenu()
end)