--[[
    CameraHotkeys

    Script adds hotkeys to the game to move the camera around the vehicle.

        @author:    BayernGamers
        @date:      05.12.2024
        @version:   1.1

        History:    v1.0 @14.11.2024 - initial implementation in FS25
                    -------------------------------------------------------------------------------------------
                    v1.1 @05.12.2024 - bugfixes and adjustments

       License:     Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.
						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]

CameraHotkeys = {}
CameraHotkeys.MOD_NAME = g_currentModName
CameraHotkeys.MOD_DIR = g_currentModDirectory
CameraHotkeys.debugLevel = 1
CameraHotkeys.debugLevelHigh = 2

source(Utils.getFilename("scripts/utils/LoggingUtil.lua", CameraHotkeys.MOD_DIR))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "CameraHotkeys.lua")

function CameraHotkeys.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Enterable, specializations)
end

function CameraHotkeys.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", CameraHotkeys)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", CameraHotkeys)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", CameraHotkeys)
    SpecializationUtil.registerEventListener(vehicleType, "onCameraChanged", CameraHotkeys)
end

function CameraHotkeys.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "resetCamera", CameraHotkeys.resetCamera)
    SpecializationUtil.registerFunction(vehicleType, "moveCamLeft", CameraHotkeys.moveCamLeft)
    SpecializationUtil.registerFunction(vehicleType, "moveCamRight", CameraHotkeys.moveCamRight)
    SpecializationUtil.registerFunction(vehicleType, "moveCamBack", CameraHotkeys.moveCamBack)
    SpecializationUtil.registerFunction(vehicleType, "moveCamFrontLeft", CameraHotkeys.moveCamFrontLeft)
    SpecializationUtil.registerFunction(vehicleType, "moveCamFrontRight", CameraHotkeys.moveCamFrontRight)
    SpecializationUtil.registerFunction(vehicleType, "toggleReverseCam", CameraHotkeys.toggleReverseCam)
end

function CameraHotkeys:onLoad(savegame)
    self.spec_cameraHotkeys = {}
    local spec = self.spec_cameraHotkeys

    spec.actionEvents = {}
    spec.resetActionEvents = {}
    spec.lastRotation = nil
    spec.lastRotationReverse = nil
    spec.reverseCamActive = false
    spec.canResetReverseCam = false
    spec.actionEventsVisible = false
    spec.outdoorCamerasDisabled = false
end

function CameraHotkeys:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    local spec = self.spec_cameraHotkeys

    if self:getIsEntered() and self.isClient then
        self:clearActionEventsTable(spec.actionEvents)
        self:clearActionEventsTable(spec.resetActionEvents)

        if isActiveForInputIgnoreSelection then
            _, camLeftActionEventId = self:addActionEvent(spec.actionEvents, InputAction.CH_CAM_LEFT, self, CameraHotkeys.moveCamLeft, false, true, false, true, nil)
            _, resetCamLeftActionEventId = self:addActionEvent(spec.resetActionEvents, InputAction.CH_CAM_LEFT, self, CameraHotkeys.resetCamera, true, false, false, true, nil)
            g_inputBinding:setActionEventTextVisibility(resetCamLeftActionEventId.actionEventId, false)
            
            _, camRightActionEventId = self:addActionEvent(spec.actionEvents, InputAction.CH_CAM_RIGHT, self, CameraHotkeys.moveCamRight, false, true, false, true, nil)
            _, resetCamRightActionEventId = self:addActionEvent(spec.resetActionEvents, InputAction.CH_CAM_RIGHT, self, CameraHotkeys.resetCamera, true, false, false, true, nil)
            g_inputBinding:setActionEventTextVisibility(resetCamRightActionEventId.actionEventId, false)
            
            _, camBackActionEventId = self:addActionEvent(spec.actionEvents, InputAction.CH_CAM_BACK, self, CameraHotkeys.moveCamBack, false, true, false, true, nil)
            _, resetCamBackActionEventId = self:addActionEvent(spec.resetActionEvents, InputAction.CH_CAM_BACK, self, CameraHotkeys.resetCamera, true, false, false, true, nil)
            g_inputBinding:setActionEventTextVisibility(resetCamBackActionEventId.actionEventId, false)

            _, camFrontLeftActionEventId = self:addActionEvent(spec.actionEvents, InputAction.CH_OUTDOOR_CAM_FRONTLEFT, self, CameraHotkeys.moveCamFrontLeft, false, true, false, true, nil)
            _, resetCamFrontLeftActionEventId = self:addActionEvent(spec.resetActionEvents, InputAction.CH_OUTDOOR_CAM_FRONTLEFT, self, CameraHotkeys.resetCamera, true, false, false, true, nil)
            g_inputBinding:setActionEventTextVisibility(resetCamFrontLeftActionEventId.actionEventId, false)

            _, camFrontRightActionEventId = self:addActionEvent(spec.actionEvents, InputAction.CH_OUTDOOR_CAM_FRONTRIGHT, self, CameraHotkeys.moveCamFrontRight, false, true, false, true, nil)
            _, resetCamFrontRightActionEventId = self:addActionEvent(spec.resetActionEvents, InputAction.CH_OUTDOOR_CAM_FRONTRIGHT, self, CameraHotkeys.resetCamera, true, false, false, true, nil)
            g_inputBinding:setActionEventTextVisibility(resetCamFrontRightActionEventId.actionEventId, false)

            _, camToggleReverseActionEventId = self:addActionEvent(spec.actionEvents, InputAction.CH_CAM_BACK_TOGGLE, self, CameraHotkeys.toggleReverseCam, false, true, false, true, nil)

            log:printDevInfo(#spec.actionEvents .. " Action events registered", CameraHotkeys.debugLevelHigh)
            log:printDevInfo(#spec.resetActionEvents .. " Reset action events registered", CameraHotkeys.debugLevelHigh)
            spec.actionEventsVisible = true
        end
    end
end

function CameraHotkeys:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    local spec = self.spec_cameraHotkeys

    if self.isClient then
        local activeCamera = self:getActiveCamera()

        if activeCamera ~= nil then
            if activeCamera.isInside and not spec.outdoorCamerasDisabled then
                moveCamFrontLeftActionEvent = spec.actionEvents[InputAction.CH_OUTDOOR_CAM_FRONTLEFT]
                moveCamFrontRightActionEvent = spec.actionEvents[InputAction.CH_OUTDOOR_CAM_FRONTRIGHT]

                if moveCamFrontLeftActionEvent ~= nil and moveCamFrontLeftActionEvent.actionEventId ~= nil then
                    g_inputBinding:setActionEventActive(moveCamFrontLeftActionEvent.actionEventId, false)
                end

                if moveCamFrontRightActionEvent ~= nil and moveCamFrontRightActionEvent.actionEventId ~= nil then
                    g_inputBinding:setActionEventActive(moveCamFrontRightActionEvent.actionEventId, false)
                end

                spec.outdoorCamerasDisabled = true
            elseif not activeCamera.isInside and spec.outdoorCamerasDisabled then
                moveCamFrontLeftActionEvent = spec.actionEvents[InputAction.CH_OUTDOOR_CAM_FRONTLEFT]
                moveCamFrontRightActionEvent = spec.actionEvents[InputAction.CH_OUTDOOR_CAM_FRONTRIGHT]

                if moveCamFrontLeftActionEvent ~= nil and moveCamFrontLeftActionEvent.actionEventId ~= nil then
                    g_inputBinding:setActionEventActive(moveCamFrontLeftActionEvent.actionEventId, true)
                end

                if moveCamFrontRightActionEvent~= nil and moveCamFrontRightActionEvent.actionEventId ~= nil then
                    g_inputBinding:setActionEventActive(moveCamFrontRightActionEvent.actionEventId, true)
                end

                spec.outdoorCamerasDisabled = false
            end

            if spec.reverseCamActive then
                camBackActionEventId = spec.actionEvents[InputAction.CH_CAM_BACK]
                if camBackActionEventId ~= nil and camBackActionEventId.actionEventId ~= nil then
                    g_inputBinding:setActionEventActive(camBackActionEventId.actionEventId, false)
                end
            elseif spec.actionEventsVisible then
                camBackActionEventId = spec.actionEvents[InputAction.CH_CAM_BACK]
                if camBackActionEventId ~= nil and camBackActionEventId.actionEventId ~= nil then
                    g_inputBinding:setActionEventActive(camBackActionEventId.actionEventId, true)
                end
            end

            if spec.reverseCamActive and not spec.canResetReverseCam then
                self:moveCamBack()
                spec.canResetReverseCam = true
            elseif not spec.reverseCamActive and spec.canResetReverseCam then
                self:resetCamera()
                spec.canResetReverseCam = false
            end

            if not activeCamera.isRotatable and spec.actionEventsVisible then
                for _, actionEvent in pairs(spec.actionEvents) do
                    if actionEvent.actionEventId ~= nil then
                        log:printDevInfo("Hiding action event " .. actionEvent.actionEventId, CameraHotkeys.debugLevelHigh)
                        g_inputBinding:setActionEventTextVisibility(actionEvent.actionEventId, false)
                        g_inputBinding:setActionEventActive(actionEvent.actionEventId, false)
                    end
                end
                for _, resetActionEvent in pairs(spec.resetActionEvents) do
                    if resetActionEvent.actionEventId ~= nil then
                        log:printDevInfo("Hiding reset action event " .. resetActionEvent.actionEventId, CameraHotkeys.debugLevelHigh)
                        g_inputBinding:setActionEventActive(resetActionEvent.actionEventId, false)
                    end
                end
                spec.actionEventsVisible = false
            elseif activeCamera.isRotatable and not spec.actionEventsVisible then
                for _, actionEvent in pairs(spec.actionEvents) do
                    if actionEvent.actionEventId ~= nil then
                        log:printDevInfo("Showing action event " .. actionEvent.actionEventId, CameraHotkeys.debugLevelHigh)
                        g_inputBinding:setActionEventTextVisibility(actionEvent.actionEventId, true)
                        g_inputBinding:setActionEventActive(actionEvent.actionEventId, true)
                    end
                end
                for _, resetActionEvent in pairs(spec.resetActionEvents) do
                    if resetActionEvent.actionEventId ~= nil then
                        log:printDevInfo("Showing reset action event " .. resetActionEvent.actionEventId, CameraHotkeys.debugLevelHigh)
                        g_inputBinding:setActionEventActive(resetActionEvent.actionEventId, true)
                    end
                end
                spec.actionEventsVisible = true
            end
        end
    end
end

function CameraHotkeys:onCameraChanged()
    local spec = self.spec_cameraHotkeys
    local activeCamera = self:getActiveCamera()

    if activeCamera ~= nil then
        log:printDevInfo("Camera changed from inside to outside or vice versa", CameraHotkeys.debugLevelHigh)
        spec.lastRotation = nil
        spec.reverseCamActive = false
        spec.canResetReverseCam = false
    end
end


function CameraHotkeys:moveCamLeft(actionName, inputValue, callbackState, isAnalog, stopPropagation)
    local spec = self.spec_cameraHotkeys
    log:printDevInfo("MoveCamLeft", CameraHotkeys.debugLevelHigh)
    
    local activeCamera = self:getActiveCamera()

    if activeCamera ~= nil then
        local rotationNode = activeCamera.rotateNode
        local rx = activeCamera.rotX
        local ry = activeCamera.rotY
        local rz = activeCamera.rotZ

        if spec.lastRotation == nil then
            log:printDevInfo("Setting Last rotation to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
            spec.lastRotation = {rx, ry, rz}
        end

        if activeCamera.isInside then
            if spec.reverseCamActive then
                if spec.lastRotationReverse == nil then
                    log:printDevInfo("Setting Last rotation reverse to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
                    spec.lastRotationReverse = {rx, ry, rz}
                end

                activeCamera.rotX = math.rad(-10)
                activeCamera.rotY = math.rad(45)
                activeCamera.rotZ = math.rad(180)
            else
                activeCamera.rotX = math.rad(-10)
                activeCamera.rotY = math.rad(-135)
                activeCamera.rotZ = math.rad(180)
            end
        else
            if spec.reverseCamActive then
                if spec.lastRotationReverse == nil then
                    log:printDevInfo("Setting Last rotation reverse to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
                    spec.lastRotationReverse = {rx, ry, rz}
                end

                if not stopPropagation or stopPropagation == nil then
                    self:moveCamFrontRight(nil, nil, nil, nil, true)
                else
                    activeCamera.rotX = math.rad(-20)
                    activeCamera.rotY = math.rad(-135)
                    activeCamera.rotZ = math.rad(180)
                end

                return
            else
                activeCamera.rotX = math.rad(-20)
                activeCamera.rotY = math.rad(-135)
                activeCamera.rotZ = math.rad(180)
            end
        end
        activeCamera:updateRotateNodeRotation()
    end
end

function CameraHotkeys:moveCamRight(actionName, inputValue, callbackState, isAnalog, stopPropagation)
    local spec = self.spec_cameraHotkeys
    log:printDevInfo("MoveCamRight", CameraHotkeys.debugLevelHigh)
    
    local activeCamera = self:getActiveCamera()

    if activeCamera ~= nil then
        local rotationNode = activeCamera.rotateNode
        local rx = activeCamera.rotX
        local ry = activeCamera.rotY
        local rz = activeCamera.rotZ

        if spec.lastRotation == nil then
            log:printDevInfo("Setting Last rotation to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
            spec.lastRotation = {rx, ry, rz}
        end

        if activeCamera.isInside then
            if spec.reverseCamActive then
                if spec.lastRotationReverse == nil then
                    log:printDevInfo("Setting Last rotation reverse to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
                    spec.lastRotationReverse = {rx, ry, rz}
                end

                activeCamera.rotX = math.rad(-10)
                activeCamera.rotY = math.rad(-45)
                activeCamera.rotZ = math.rad(180)
            else
                activeCamera.rotX = math.rad(-10)
                activeCamera.rotY = math.rad(135)
                activeCamera.rotZ = math.rad(180)
            end
        else
            if spec.reverseCamActive then
                if spec.lastRotationReverse == nil then
                    log:printDevInfo("Setting Last rotation reverse to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
                    spec.lastRotationReverse = {rx, ry, rz}
                end

                if not stopPropagation or stopPropagation == nil then
                    self:moveCamFrontLeft(nil, nil, nil, nil, true)
                else
                    activeCamera.rotX = math.rad(-20)
                    activeCamera.rotY = math.rad(135)
                    activeCamera.rotZ = math.rad(180)
                end

                return
            else
                activeCamera.rotX = math.rad(-20)
                activeCamera.rotY = math.rad(135)
                activeCamera.rotZ = math.rad(180)
            end
        end
        activeCamera:updateRotateNodeRotation()
    end
end

function CameraHotkeys:moveCamBack(actionName, inputValue, callbackState, isAnalog, stopPropagation)
    local spec = self.spec_cameraHotkeys
    log:printDevInfo("MoveCamBack", CameraHotkeys.debugLevelHigh)
    
    local activeCamera = self:getActiveCamera()

    if activeCamera ~= nil then
        local rotationNode = activeCamera.rotateNode
        local rx = activeCamera.rotX
        local ry = activeCamera.rotY
        local rz = activeCamera.rotZ

        if spec.lastRotation == nil then
            log:printDevInfo("Setting Last rotation to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
            spec.lastRotation = {rx, ry, rz}
        end

        if activeCamera.isInside then
            activeCamera.rotX = math.rad(-15)
            activeCamera.rotY = math.rad(0)
            activeCamera.rotZ = math.rad(180)
        else
            activeCamera.rotX = math.rad(-20)
            activeCamera.rotY = math.rad(0)
            activeCamera.rotZ = math.rad(180)
        end
        activeCamera:updateRotateNodeRotation()
    end
end

function CameraHotkeys:moveCamFrontLeft(actionName, inputValue, callbackState, isAnalog, stopPropagation)
    local spec = self.spec_cameraHotkeys
    local activeCamera = self:getActiveCamera()

    if activeCamera ~= nil then
        log:printDevInfo("MoveCamFrontLeft", CameraHotkeys.debugLevelHigh)

        if activeCamera.isInside then
            log:printDevInfo("Camera is inside", CameraHotkeys.debugLevelHigh)
            return
        end
        
        local rotationNode = activeCamera.rotateNode
        local rx = activeCamera.rotX
        local ry = activeCamera.rotY
        local rz = activeCamera.rotZ

        if spec.lastRotation == nil then
            log:printDevInfo("Setting Last rotation to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
            spec.lastRotation = {rx, ry, rz}
        end

        if spec.reverseCamActive then
            --TODO: Fix Infinite Loop
            if spec.lastRotationReverse == nil then
                log:printDevInfo("Setting Last rotation reverse to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
                spec.lastRotationReverse = {rx, ry, rz}
            end

            if not stopPropagation or stopPropagation == nil then
                self:moveCamLeft(nil, nil, nil, nil, true)
            else
                activeCamera.rotX = math.rad(-20)
                activeCamera.rotY = math.rad(-45)
                activeCamera.rotZ = math.rad(180)
            end

            return
        else
            activeCamera.rotX = math.rad(-20)
            activeCamera.rotY = math.rad(45)
            activeCamera.rotZ = math.rad(180)
        end

        activeCamera:updateRotateNodeRotation()
    end
end

function CameraHotkeys:moveCamFrontRight(actionName, inputValue, callbackState, isAnalog, stopPropagation)
    local spec = self.spec_cameraHotkeys
    local activeCamera = self:getActiveCamera()

    if activeCamera ~= nil then
        log:printDevInfo("MoveCamFrontRight", CameraHotkeys.debugLevelHigh)

        if activeCamera.isInside then
            log:printDevInfo("Camera is inside", CameraHotkeys.debugLevelHigh)
            return
        end
        
        local rotationNode = activeCamera.rotateNode
        local rx = activeCamera.rotX
        local ry = activeCamera.rotY
        local rz = activeCamera.rotZ

        if spec.lastRotation == nil then
            log:printDevInfo("Setting Last rotation to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
            spec.lastRotation = {rx, ry, rz}
        end

        if spec.reverseCamActive then
            --TODO: Fix Infinite Loop
            if spec.lastRotationReverse == nil then
                log:printDevInfo("Setting Last rotation reverse to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
                spec.lastRotationReverse = {rx, ry, rz}
            end

            if not stopPropagation or stopPropagation == nil then
                self:moveCamRight(nil, nil, nil, nil, true)
            else
                activeCamera.rotX = math.rad(-20)
                activeCamera.rotY = math.rad(45)
                activeCamera.rotZ = math.rad(180)
            end

            return
        else
            activeCamera.rotX = math.rad(-20)
            activeCamera.rotY = math.rad(-45)
            activeCamera.rotZ = math.rad(180)
        end

        activeCamera:updateRotateNodeRotation()
    end
end

function CameraHotkeys:toggleReverseCam(actionName, inputValue, callbackState, isAnalog)
    local spec = self.spec_cameraHotkeys
    log:printDevInfo("ToggleReverseCam", CameraHotkeys.debugLevelHigh)

    spec.reverseCamActive = not spec.reverseCamActive
    local activeCamera = self:getActiveCamera()

    if activeCamera ~= nil then
        local actionEvent = spec.actionEvents[InputAction.CH_CAM_BACK_TOGGLE]
        local actionEventId = actionEvent.actionEventId
        log:printDevInfo("ActionEventId: " .. actionEventId, CameraHotkeys.debugLevelHigh)

        if spec.reverseCamActive then
            log:printDevInfo("Reverse cam is active. Setting Text", CameraHotkeys.debugLevelHigh)
            g_inputBinding:setActionEventText(actionEventId, g_i18n:getText("action_CH_CAM_BACK_DISABLE"))
        else
            log:printDevInfo("Reverse cam is inactive. Setting Text.", CameraHotkeys.debugLevelHigh)
            g_inputBinding:setActionEventText(actionEventId, g_i18n:getText("action_CH_CAM_BACK_ENABLE"))
        end
    end
end

function CameraHotkeys:resetCamera(actionName, inputValue, callbackState, isAnalog)
    local spec = self.spec_cameraHotkeys
    log:printDevInfo("ResetCamera", CameraHotkeys.debugLevelHigh)

    log:printDevInfo("LastRotation: " .. tostring(spec.lastRotation), CameraHotkeys.debugLevelHigh)
    log:printDevInfo("LastRotationReverse: " .. tostring(spec.lastRotationReverse), CameraHotkeys.debugLevelHigh)
    log:printDevInfo("ReverseCamActive: " .. tostring(spec.reverseCamActive), CameraHotkeys.debugLevelHigh)
    
    if spec.lastRotation ~= nil and not spec.reverseCamActive then
        local activeCamera = self:getActiveCamera()

        if activeCamera ~= nil then
            local rotationNode = activeCamera.rotateNode
            local rx, ry, rz = unpack(spec.lastRotation)
            log:printDevInfo("Setting rotation to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
            activeCamera.rotX = rx
            activeCamera.rotY = ry
            activeCamera.rotZ = rz
            activeCamera:updateRotateNodeRotation()
            spec.lastRotation = nil
        end
    elseif spec.lastRotationReverse ~= nil and spec.reverseCamActive then
        local activeCamera = self:getActiveCamera()

        if activeCamera ~= nil then
            local rotationNode = activeCamera.rotateNode
            local rx, ry, rz = unpack(spec.lastRotationReverse)
            log:printDevInfo("Setting rotation from rotationReverse to " .. rx .. ", " .. ry .. ", " .. rz, CameraHotkeys.debugLevelHigh)
            activeCamera.rotX = rx
            activeCamera.rotY = ry
            activeCamera.rotZ = rz
            activeCamera:updateRotateNodeRotation()
            spec.lastRotationReverse = nil
        end
    end
end